-- ============================================================
-- OptiCore SaaS - Script de Instalación Completo
-- Ejecutar este archivo para crear la BD desde cero
-- MySQL 8.0 | Versión 1.0.0
-- ============================================================

SET FOREIGN_KEY_CHECKS = 0;
SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";

-- ── Crear y usar la base de datos ─────────────────────────────
DROP DATABASE IF EXISTS `opticore`;
CREATE DATABASE `opticore`
  CHARACTER SET utf8mb4
  COLLATE utf8mb4_unicode_ci;

USE `opticore`;

-- ============================================================
-- TABLAS
-- ============================================================

-- 1. planes
CREATE TABLE `planes` (
  `id`                INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `nombre`            VARCHAR(100) NOT NULL,
  `descripcion`       TEXT,
  `precio_mensual`    DECIMAL(10,2) NOT NULL DEFAULT 0.00,
  `precio_anual`      DECIMAL(10,2) NOT NULL DEFAULT 0.00,
  `max_sucursales`    INT NOT NULL DEFAULT 1,
  `max_usuarios`      INT NOT NULL DEFAULT 5,
  `max_productos`     INT NOT NULL DEFAULT 100,
  `max_pacientes`     INT NOT NULL DEFAULT 500,
  `tiene_laboratorio` TINYINT(1) NOT NULL DEFAULT 0,
  `tiene_agenda`      TINYINT(1) NOT NULL DEFAULT 0,
  `tiene_reportes`    TINYINT(1) NOT NULL DEFAULT 1,
  `tiene_api`         TINYINT(1) NOT NULL DEFAULT 0,
  `estado`            ENUM('activo','inactivo') NOT NULL DEFAULT 'activo',
  `created_at`        TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`        TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 2. empresas
CREATE TABLE `empresas` (
  `id`             INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `plan_id`        INT UNSIGNED NOT NULL,
  `nombre`         VARCHAR(200) NOT NULL,
  `razon_social`   VARCHAR(200),
  `rut`            VARCHAR(20),
  `email`          VARCHAR(150),
  `telefono`       VARCHAR(30),
  `direccion`      VARCHAR(300),
  `ciudad`         VARCHAR(100),
  `pais`           VARCHAR(100) DEFAULT 'Chile',
  `moneda`         VARCHAR(10) DEFAULT 'CLP',
  `logo`           VARCHAR(300),
  `color_primario` VARCHAR(10) DEFAULT '#0d6efd',
  `estado`         ENUM('activa','suspendida','expirada','prueba') NOT NULL DEFAULT 'prueba',
  `created_at`     TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`     TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_plan_id` (`plan_id`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_empresas_plan` FOREIGN KEY (`plan_id`) REFERENCES `planes` (`id`) ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 3. licencias
CREATE TABLE `licencias` (
  `id`            INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`    INT UNSIGNED NOT NULL,
  `plan_id`       INT UNSIGNED NOT NULL,
  `tipo`          ENUM('mensual','anual','perpetua','prueba') NOT NULL DEFAULT 'mensual',
  `fecha_inicio`  DATE NOT NULL,
  `fecha_fin`     DATE,
  `monto`         DECIMAL(10,2) NOT NULL DEFAULT 0.00,
  `estado`        ENUM('activa','suspendida','expirada','cancelada') NOT NULL DEFAULT 'activa',
  `observaciones` TEXT,
  `created_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_plan_id` (`plan_id`),
  KEY `idx_estado` (`estado`),
  KEY `idx_fecha_fin` (`fecha_fin`),
  CONSTRAINT `fk_licencias_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_licencias_plan` FOREIGN KEY (`plan_id`) REFERENCES `planes` (`id`) ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 4. sucursales
CREATE TABLE `sucursales` (
  `id`                    INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`            INT UNSIGNED NOT NULL,
  `nombre`                VARCHAR(150) NOT NULL,
  `codigo`                VARCHAR(20),
  `direccion`             VARCHAR(300),
  `telefono`              VARCHAR(30),
  `email`                 VARCHAR(150),
  `horario`               VARCHAR(200),
  `correlativo_ventas`    INT UNSIGNED NOT NULL DEFAULT 1,
  `correlativo_ordenes`   INT UNSIGNED NOT NULL DEFAULT 1,
  `correlativo_caja`      INT UNSIGNED NOT NULL DEFAULT 1,
  `es_principal`          TINYINT(1) NOT NULL DEFAULT 0,
  `estado`                ENUM('activa','inactiva') NOT NULL DEFAULT 'activa',
  `created_at`            TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`            TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_sucursales_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 5. roles
CREATE TABLE `roles` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`  INT UNSIGNED,
  `nombre`      VARCHAR(100) NOT NULL,
  `slug`        VARCHAR(100) NOT NULL,
  `descripcion` TEXT,
  `es_sistema`  TINYINT(1) NOT NULL DEFAULT 0,
  `estado`      ENUM('activo','inactivo') NOT NULL DEFAULT 'activo',
  `created_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_slug` (`slug`),
  CONSTRAINT `fk_roles_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 6. permisos
CREATE TABLE `permisos` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `modulo`      VARCHAR(100) NOT NULL,
  `nombre`      VARCHAR(100) NOT NULL,
  `descripcion` VARCHAR(200),
  `created_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_nombre` (`nombre`),
  KEY `idx_modulo` (`modulo`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 7. rol_permisos
CREATE TABLE `rol_permisos` (
  `rol_id`     INT UNSIGNED NOT NULL,
  `permiso_id` INT UNSIGNED NOT NULL,
  `granted`    TINYINT(1) NOT NULL DEFAULT 1,
  PRIMARY KEY (`rol_id`, `permiso_id`),
  KEY `idx_permiso_id` (`permiso_id`),
  CONSTRAINT `fk_rp_rol` FOREIGN KEY (`rol_id`) REFERENCES `roles` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_rp_permiso` FOREIGN KEY (`permiso_id`) REFERENCES `permisos` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 8. usuarios
CREATE TABLE `usuarios` (
  `id`            INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`    INT UNSIGNED,
  `sucursal_id`   INT UNSIGNED,
  `rol_id`        INT UNSIGNED NOT NULL,
  `nombre`        VARCHAR(100) NOT NULL,
  `apellido`      VARCHAR(100) NOT NULL,
  `email`         VARCHAR(150) NOT NULL,
  `password`      VARCHAR(255) NOT NULL,
  `telefono`      VARCHAR(30),
  `avatar`        VARCHAR(300),
  `es_superadmin` TINYINT(1) NOT NULL DEFAULT 0,
  `estado`        ENUM('activo','inactivo','bloqueado') NOT NULL DEFAULT 'activo',
  `ultimo_login`  TIMESTAMP NULL,
  `token_reset`   VARCHAR(100),
  `token_expira`  TIMESTAMP NULL,
  `created_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_email` (`email`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sucursal_id` (`sucursal_id`),
  KEY `idx_rol_id` (`rol_id`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_usuarios_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_usuarios_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_usuarios_rol` FOREIGN KEY (`rol_id`) REFERENCES `roles` (`id`) ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 9. pacientes
CREATE TABLE `pacientes` (
  `id`               INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`       INT UNSIGNED NOT NULL,
  `sucursal_id`      INT UNSIGNED,
  `nombre`           VARCHAR(100) NOT NULL,
  `apellido`         VARCHAR(100) NOT NULL,
  `rut`              VARCHAR(20),
  `documento`        VARCHAR(30),
  `fecha_nacimiento` DATE,
  `genero`           ENUM('M','F','otro'),
  `telefono`         VARCHAR(30),
  `email`            VARCHAR(150),
  `direccion`        VARCHAR(300),
  `ciudad`           VARCHAR(100),
  `ocupacion`        VARCHAR(100),
  `observaciones`    TEXT,
  `estado`           ENUM('activo','inactivo') NOT NULL DEFAULT 'activo',
  `created_at`       TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`       TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sucursal_id` (`sucursal_id`),
  KEY `idx_nombre` (`nombre`, `apellido`),
  KEY `idx_rut` (`rut`),
  CONSTRAINT `fk_pacientes_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_pacientes_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 10. recetas
CREATE TABLE `recetas` (
  `id`           INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`   INT UNSIGNED NOT NULL,
  `sucursal_id`  INT UNSIGNED,
  `paciente_id`  INT UNSIGNED NOT NULL,
  `usuario_id`   INT UNSIGNED NOT NULL,
  `fecha`        DATE NOT NULL,
  `vigencia`     DATE,
  `tipo`         ENUM('cerca','lejos','progresivo','bifocal') NOT NULL DEFAULT 'lejos',
  `od_esfera`    DECIMAL(5,2),
  `od_cilindro`  DECIMAL(5,2),
  `od_eje`       INT,
  `od_adicion`   DECIMAL(5,2),
  `od_dp`        DECIMAL(5,2),
  `od_av`        VARCHAR(20),
  `oi_esfera`    DECIMAL(5,2),
  `oi_cilindro`  DECIMAL(5,2),
  `oi_eje`       INT,
  `oi_adicion`   DECIMAL(5,2),
  `oi_dp`        DECIMAL(5,2),
  `oi_av`        VARCHAR(20),
  `dp_total`     DECIMAL(5,2),
  `observaciones` TEXT,
  `adjunto`      VARCHAR(300),
  `estado`       ENUM('activa','vencida','anulada') NOT NULL DEFAULT 'activa',
  `created_at`   TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`   TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_paciente_id` (`paciente_id`),
  KEY `idx_usuario_id` (`usuario_id`),
  KEY `idx_fecha` (`fecha`),
  CONSTRAINT `fk_recetas_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_recetas_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_recetas_paciente` FOREIGN KEY (`paciente_id`) REFERENCES `pacientes` (`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_recetas_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 11. categorias
CREATE TABLE `categorias` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`  INT UNSIGNED NOT NULL,
  `nombre`      VARCHAR(100) NOT NULL,
  `descripcion` TEXT,
  `estado`      ENUM('activa','inactiva') NOT NULL DEFAULT 'activa',
  `created_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  CONSTRAINT `fk_categorias_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 12. marcas
CREATE TABLE `marcas` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`  INT UNSIGNED NOT NULL,
  `nombre`      VARCHAR(100) NOT NULL,
  `descripcion` TEXT,
  `estado`      ENUM('activa','inactiva') NOT NULL DEFAULT 'activa',
  `created_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  CONSTRAINT `fk_marcas_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 13. proveedores
CREATE TABLE `proveedores` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`  INT UNSIGNED NOT NULL,
  `nombre`      VARCHAR(200) NOT NULL,
  `rut`         VARCHAR(20),
  `contacto`    VARCHAR(100),
  `telefono`    VARCHAR(30),
  `email`       VARCHAR(150),
  `direccion`   VARCHAR(300),
  `tipo`        ENUM('proveedor','laboratorio','ambos') NOT NULL DEFAULT 'proveedor',
  `estado`      ENUM('activo','inactivo') NOT NULL DEFAULT 'activo',
  `created_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  CONSTRAINT `fk_proveedores_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 14. productos
CREATE TABLE `productos` (
  `id`            INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`    INT UNSIGNED NOT NULL,
  `categoria_id`  INT UNSIGNED,
  `marca_id`      INT UNSIGNED,
  `proveedor_id`  INT UNSIGNED,
  `sku`           VARCHAR(100),
  `codigo_barras` VARCHAR(100),
  `nombre`        VARCHAR(200) NOT NULL,
  `descripcion`   TEXT,
  `tipo`          ENUM('marco','lente','solucion','accesorio','servicio','otro') NOT NULL DEFAULT 'otro',
  `precio_costo`  DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `precio_venta`  DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `impuesto`      DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  `stock_minimo`  INT NOT NULL DEFAULT 0,
  `imagen`        VARCHAR(300),
  `estado`        ENUM('activo','inactivo') NOT NULL DEFAULT 'activo',
  `created_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_categoria_id` (`categoria_id`),
  KEY `idx_marca_id` (`marca_id`),
  KEY `idx_sku` (`sku`),
  KEY `idx_tipo` (`tipo`),
  CONSTRAINT `fk_productos_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_productos_categoria` FOREIGN KEY (`categoria_id`) REFERENCES `categorias` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_productos_marca` FOREIGN KEY (`marca_id`) REFERENCES `marcas` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_productos_proveedor` FOREIGN KEY (`proveedor_id`) REFERENCES `proveedores` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 15. stock
CREATE TABLE `stock` (
  `id`          INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`  INT UNSIGNED NOT NULL,
  `sucursal_id` INT UNSIGNED NOT NULL,
  `producto_id` INT UNSIGNED NOT NULL,
  `cantidad`    INT NOT NULL DEFAULT 0,
  `updated_at`  TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_sucursal_producto` (`sucursal_id`, `producto_id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_producto_id` (`producto_id`),
  CONSTRAINT `fk_stock_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_stock_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_stock_producto` FOREIGN KEY (`producto_id`) REFERENCES `productos` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 16. movimientos_stock
CREATE TABLE `movimientos_stock` (
  `id`                INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`        INT UNSIGNED NOT NULL,
  `sucursal_id`       INT UNSIGNED NOT NULL,
  `producto_id`       INT UNSIGNED NOT NULL,
  `usuario_id`        INT UNSIGNED,
  `tipo`              ENUM('entrada','salida','ajuste','traspaso_entrada','traspaso_salida','venta','devolucion') NOT NULL,
  `cantidad`          INT NOT NULL,
  `cantidad_anterior` INT NOT NULL DEFAULT 0,
  `cantidad_nueva`    INT NOT NULL DEFAULT 0,
  `costo_unitario`    DECIMAL(12,2) DEFAULT 0.00,
  `referencia_tipo`   VARCHAR(50),
  `referencia_id`     INT UNSIGNED,
  `observaciones`     TEXT,
  `fecha`             TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sucursal_id` (`sucursal_id`),
  KEY `idx_producto_id` (`producto_id`),
  KEY `idx_tipo` (`tipo`),
  KEY `idx_fecha` (`fecha`),
  CONSTRAINT `fk_mov_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_mov_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_mov_producto` FOREIGN KEY (`producto_id`) REFERENCES `productos` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 17. formas_pago
CREATE TABLE `formas_pago` (
  `id`         INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id` INT UNSIGNED NOT NULL,
  `nombre`     VARCHAR(100) NOT NULL,
  `tipo`       ENUM('efectivo','tarjeta','transferencia','cheque','credito','otro') NOT NULL DEFAULT 'efectivo',
  `estado`     ENUM('activa','inactiva') NOT NULL DEFAULT 'activa',
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  CONSTRAINT `fk_fp_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 18. ventas
CREATE TABLE `ventas` (
  `id`            INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`    INT UNSIGNED NOT NULL,
  `sucursal_id`   INT UNSIGNED NOT NULL,
  `usuario_id`    INT UNSIGNED NOT NULL,
  `paciente_id`   INT UNSIGNED,
  `correlativo`   VARCHAR(20) NOT NULL,
  `fecha`         TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `subtotal`      DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `descuento`     DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `impuesto`      DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `total`         DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `estado`        ENUM('cotizacion','confirmada','anulada') NOT NULL DEFAULT 'confirmada',
  `observaciones` TEXT,
  `created_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`    TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sucursal_id` (`sucursal_id`),
  KEY `idx_usuario_id` (`usuario_id`),
  KEY `idx_paciente_id` (`paciente_id`),
  KEY `idx_correlativo` (`correlativo`),
  KEY `idx_fecha` (`fecha`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_ventas_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_ventas_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_ventas_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_ventas_paciente` FOREIGN KEY (`paciente_id`) REFERENCES `pacientes` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 19. venta_items
CREATE TABLE `venta_items` (
  `id`              INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `venta_id`        INT UNSIGNED NOT NULL,
  `producto_id`     INT UNSIGNED NOT NULL,
  `nombre_producto` VARCHAR(200) NOT NULL,
  `cantidad`        INT NOT NULL DEFAULT 1,
  `precio_unitario` DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `descuento`       DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `impuesto`        DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `subtotal`        DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  PRIMARY KEY (`id`),
  KEY `idx_venta_id` (`venta_id`),
  KEY `idx_producto_id` (`producto_id`),
  CONSTRAINT `fk_vi_venta` FOREIGN KEY (`venta_id`) REFERENCES `ventas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_vi_producto` FOREIGN KEY (`producto_id`) REFERENCES `productos` (`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 20. venta_pagos
CREATE TABLE `venta_pagos` (
  `id`            INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `venta_id`      INT UNSIGNED NOT NULL,
  `forma_pago_id` INT UNSIGNED NOT NULL,
  `monto`         DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `referencia`    VARCHAR(100),
  PRIMARY KEY (`id`),
  KEY `idx_venta_id` (`venta_id`),
  KEY `idx_forma_pago_id` (`forma_pago_id`),
  CONSTRAINT `fk_vp_venta` FOREIGN KEY (`venta_id`) REFERENCES `ventas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_vp_forma_pago` FOREIGN KEY (`forma_pago_id`) REFERENCES `formas_pago` (`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 21. caja_sesiones
CREATE TABLE `caja_sesiones` (
  `id`                    INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`            INT UNSIGNED NOT NULL,
  `sucursal_id`           INT UNSIGNED NOT NULL,
  `usuario_id`            INT UNSIGNED NOT NULL,
  `correlativo`           VARCHAR(20),
  `fecha_apertura`        TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `monto_apertura`        DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `fecha_cierre`          TIMESTAMP NULL,
  `monto_cierre_real`     DECIMAL(12,2),
  `monto_cierre_sistema`  DECIMAL(12,2),
  `diferencia`            DECIMAL(12,2),
  `estado`                ENUM('abierta','cerrada') NOT NULL DEFAULT 'abierta',
  `observaciones`         TEXT,
  `created_at`            TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sucursal_id` (`sucursal_id`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_cs_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_cs_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_cs_usuario` FOREIGN KEY (`usuario_id`) REFERENCES `usuarios` (`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 22. caja_movimientos
CREATE TABLE `caja_movimientos` (
  `id`               INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`       INT UNSIGNED NOT NULL,
  `sucursal_id`      INT UNSIGNED NOT NULL,
  `sesion_id`        INT UNSIGNED NOT NULL,
  `usuario_id`       INT UNSIGNED,
  `tipo`             ENUM('ingreso','egreso') NOT NULL,
  `concepto`         VARCHAR(200) NOT NULL,
  `monto`            DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  `forma_pago_id`    INT UNSIGNED,
  `referencia_tipo`  VARCHAR(50),
  `referencia_id`    INT UNSIGNED,
  `fecha`            TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_sesion_id` (`sesion_id`),
  CONSTRAINT `fk_cm_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_cm_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_cm_sesion` FOREIGN KEY (`sesion_id`) REFERENCES `caja_sesiones` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 23. ordenes_laboratorio
CREATE TABLE `ordenes_laboratorio` (
  `id`              INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`      INT UNSIGNED NOT NULL,
  `sucursal_id`     INT UNSIGNED NOT NULL,
  `venta_id`        INT UNSIGNED,
  `receta_id`       INT UNSIGNED,
  `paciente_id`     INT UNSIGNED,
  `proveedor_id`    INT UNSIGNED,
  `usuario_id`      INT UNSIGNED NOT NULL,
  `correlativo`     VARCHAR(20),
  `estado`          ENUM('recibido','en_proceso','listo','entregado','cancelado') NOT NULL DEFAULT 'recibido',
  `fecha_recepcion` DATE NOT NULL,
  `fecha_estimada`  DATE,
  `fecha_entrega`   DATE,
  `costo`           DECIMAL(12,2) DEFAULT 0.00,
  `descripcion`     TEXT,
  `observaciones`   TEXT,
  `created_at`      TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`      TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_ol_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_ol_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE RESTRICT,
  CONSTRAINT `fk_ol_venta` FOREIGN KEY (`venta_id`) REFERENCES `ventas` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_ol_receta` FOREIGN KEY (`receta_id`) REFERENCES `recetas` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_ol_proveedor` FOREIGN KEY (`proveedor_id`) REFERENCES `proveedores` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 24. citas
CREATE TABLE `citas` (
  `id`              INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`      INT UNSIGNED NOT NULL,
  `sucursal_id`     INT UNSIGNED NOT NULL,
  `paciente_id`     INT UNSIGNED,
  `usuario_id`      INT UNSIGNED,
  `fecha`           DATE NOT NULL,
  `hora`            TIME NOT NULL,
  `duracion`        INT NOT NULL DEFAULT 30,
  `tipo`            ENUM('optometria','entrega','control','otro') NOT NULL DEFAULT 'optometria',
  `estado`          ENUM('pendiente','confirmada','atendida','cancelada','no_asistio') NOT NULL DEFAULT 'pendiente',
  `nombre_paciente` VARCHAR(200),
  `telefono`        VARCHAR(30),
  `observaciones`   TEXT,
  `created_at`      TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at`      TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_fecha` (`fecha`),
  KEY `idx_estado` (`estado`),
  CONSTRAINT `fk_citas_empresa` FOREIGN KEY (`empresa_id`) REFERENCES `empresas` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_citas_sucursal` FOREIGN KEY (`sucursal_id`) REFERENCES `sucursales` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_citas_paciente` FOREIGN KEY (`paciente_id`) REFERENCES `pacientes` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 25. audit_logs
CREATE TABLE `audit_logs` (
  `id`               INT UNSIGNED NOT NULL AUTO_INCREMENT,
  `empresa_id`       INT UNSIGNED,
  `usuario_id`       INT UNSIGNED,
  `accion`           VARCHAR(100) NOT NULL,
  `tabla`            VARCHAR(100),
  `registro_id`      INT UNSIGNED,
  `datos_anteriores` JSON,
  `datos_nuevos`     JSON,
  `ip`               VARCHAR(45),
  `user_agent`       VARCHAR(300),
  `fecha`            TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_empresa_id` (`empresa_id`),
  KEY `idx_usuario_id` (`usuario_id`),
  KEY `idx_accion` (`accion`),
  KEY `idx_fecha` (`fecha`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

SET FOREIGN_KEY_CHECKS = 1;

-- ============================================================
-- DATOS INICIALES
-- ============================================================

-- ── Planes ───────────────────────────────────────────────────
INSERT INTO `planes` (`id`, `nombre`, `descripcion`, `precio_mensual`, `precio_anual`, `max_sucursales`, `max_usuarios`, `max_productos`, `max_pacientes`, `tiene_laboratorio`, `tiene_agenda`, `tiene_reportes`, `tiene_api`, `estado`) VALUES
(1, 'Básico',      'Plan para ópticas pequeñas, 1 sucursal',         29900, 299000,  1,  5,  200,  500, 0, 0, 1, 0, 'activo'),
(2, 'Profesional', 'Plan para ópticas medianas, hasta 3 sucursales', 59900, 599000,  3, 15, 1000, 2000, 1, 1, 1, 0, 'activo'),
(3, 'Enterprise',  'Plan para cadenas, sucursales ilimitadas',        99900, 999000, 99, 99, 9999, 9999, 1, 1, 1, 1, 'activo');

-- ── Roles de sistema ─────────────────────────────────────────
INSERT INTO `roles` (`id`, `empresa_id`, `nombre`, `slug`, `descripcion`, `es_sistema`, `estado`) VALUES
(1, NULL, 'Super Administrador', 'superadmin',    'Acceso total al SaaS',            1, 'activo'),
(2, NULL, 'Admin Empresa',       'admin_empresa', 'Administrador de empresa',        1, 'activo'),
(3, NULL, 'Jefe de Sucursal',    'jefe_sucursal', 'Gestiona su sucursal',            1, 'activo'),
(4, NULL, 'Vendedor',            'vendedor',      'Ventas y clientes',               1, 'activo'),
(5, NULL, 'Optometrista',        'optometrista',  'Recetas, pacientes y agenda',     1, 'activo'),
(6, NULL, 'Cajero',              'cajero',        'Apertura/cierre de caja y pagos', 1, 'activo');

-- ── Permisos (formato: modulo.accion) ────────────────────────
INSERT INTO `permisos` (`modulo`, `nombre`, `descripcion`) VALUES
('dashboard',   'dashboard.ver',             'Ver panel principal'),
('saas',        'saas.empresas.ver',          'Ver listado de empresas'),
('saas',        'saas.empresas.crear',        'Crear nuevas empresas'),
('saas',        'saas.empresas.editar',       'Editar empresas'),
('saas',        'saas.empresas.eliminar',     'Eliminar empresas'),
('saas',        'saas.planes.ver',            'Ver planes SaaS'),
('saas',        'saas.planes.gestionar',      'Crear/editar planes'),
('saas',        'saas.licencias.ver',         'Ver licencias'),
('saas',        'saas.licencias.gestionar',   'Crear/editar licencias'),
('sucursales',  'sucursales.ver',             'Ver sucursales'),
('sucursales',  'sucursales.crear',           'Crear sucursales'),
('sucursales',  'sucursales.editar',          'Editar sucursales'),
('sucursales',  'sucursales.eliminar',        'Eliminar sucursales'),
('usuarios',    'usuarios.ver',               'Ver usuarios'),
('usuarios',    'usuarios.crear',             'Crear usuarios'),
('usuarios',    'usuarios.editar',            'Editar usuarios'),
('usuarios',    'usuarios.eliminar',          'Eliminar usuarios'),
('roles',       'roles.ver',                  'Ver roles y permisos'),
('roles',       'roles.crear',                'Crear roles'),
('roles',       'roles.editar',               'Editar roles'),
('roles',       'roles.eliminar',             'Eliminar roles'),
('pacientes',   'pacientes.ver',              'Ver pacientes'),
('pacientes',   'pacientes.crear',            'Crear pacientes'),
('pacientes',   'pacientes.editar',           'Editar pacientes'),
('pacientes',   'pacientes.eliminar',         'Eliminar pacientes'),
('recetas',     'recetas.ver',                'Ver recetas'),
('recetas',     'recetas.crear',              'Crear recetas'),
('recetas',     'recetas.editar',             'Editar recetas'),
('recetas',     'recetas.imprimir',           'Imprimir/exportar recetas'),
('productos',   'productos.ver',              'Ver productos'),
('productos',   'productos.crear',            'Crear productos'),
('productos',   'productos.editar',           'Editar productos'),
('productos',   'productos.eliminar',         'Eliminar productos'),
('stock',       'stock.ver',                  'Ver stock e inventario'),
('stock',       'stock.ajustar',              'Ajustar stock'),
('stock',       'stock.transferir',           'Transferir stock entre sucursales'),
('stock',       'stock.kardex',               'Ver Kardex de Stock'),
('ventas',      'ventas.ver',                 'Ver ventas'),
('ventas',      'ventas.crear',               'Crear ventas (POS)'),
('ventas',      'ventas.editar',              'Editar ventas'),
('ventas',      'ventas.anular',              'Anular ventas'),
('ventas',      'ventas.imprimir',            'Imprimir/exportar ventas'),
('ventas',      'ventas.cotizaciones',        'Ver cotizaciones'),
('caja',        'caja.ver',                   'Ver caja'),
('caja',        'caja.abrir',                 'Abrir sesión de caja'),
('caja',        'caja.cerrar',                'Cerrar sesión de caja'),
('caja',        'caja.movimiento',            'Ver/registrar movimientos de caja'),
('laboratorio', 'laboratorio.ver',            'Ver órdenes de laboratorio'),
('laboratorio', 'laboratorio.crear',          'Crear órdenes de laboratorio'),
('laboratorio', 'laboratorio.editar',         'Editar órdenes de laboratorio'),
('citas',       'citas.ver',                  'Ver agenda de citas'),
('citas',       'citas.crear',                'Crear citas'),
('citas',       'citas.editar',               'Editar citas'),
('reportes',    'reportes.ver',               'Ver reportes'),
('reportes',    'reportes.ventas',            'Reporte de Ventas'),
('reportes',    'reportes.stock',             'Reporte de Stock'),
('reportes',    'reportes.caja',              'Reporte de Caja'),
('reportes',    'reportes.exportar',          'Exportar reportes PDF/Excel'),
('empresa',     'empresa.ver',                'Ver configuración empresa'),
('empresa',     'empresa.editar',             'Editar configuración empresa'),
('proveedores', 'proveedores.ver',            'Ver proveedores'),
('proveedores', 'proveedores.gestionar',      'Gestionar proveedores'),
('compras',     'compras.ver',                'Ver compras'),
('documentos',  'documentos.ver',             'Ver documentos');

-- ── Asignación de permisos a roles ───────────────────────────

-- superadmin: todos los permisos
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 1, id, 1 FROM `permisos`;

-- admin_empresa: todo excepto SaaS
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 2, id, 1 FROM `permisos` WHERE `nombre` NOT LIKE 'saas.%';

-- jefe_sucursal
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 3, id, 1 FROM `permisos` WHERE `nombre` IN (
  'dashboard.ver','sucursales.ver','usuarios.ver',
  'pacientes.ver','pacientes.crear','pacientes.editar',
  'recetas.ver','recetas.crear','recetas.editar','recetas.imprimir',
  'productos.ver','stock.ver','stock.ajustar','stock.transferir','stock.kardex',
  'ventas.ver','ventas.crear','ventas.editar','ventas.anular','ventas.imprimir','ventas.cotizaciones',
  'caja.ver','caja.abrir','caja.cerrar','caja.movimiento',
  'laboratorio.ver','laboratorio.crear','laboratorio.editar',
  'citas.ver','citas.crear','citas.editar',
  'reportes.ver','reportes.ventas','reportes.stock','reportes.caja','reportes.exportar',
  'proveedores.ver','compras.ver','documentos.ver'
);

-- vendedor
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 4, id, 1 FROM `permisos` WHERE `nombre` IN (
  'dashboard.ver',
  'pacientes.ver','pacientes.crear','pacientes.editar',
  'recetas.ver','recetas.imprimir',
  'productos.ver','stock.ver',
  'ventas.ver','ventas.crear','ventas.imprimir','ventas.cotizaciones',
  'caja.ver','citas.ver','documentos.ver'
);

-- optometrista
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 5, id, 1 FROM `permisos` WHERE `nombre` IN (
  'dashboard.ver',
  'pacientes.ver','pacientes.crear','pacientes.editar',
  'recetas.ver','recetas.crear','recetas.editar','recetas.imprimir',
  'citas.ver','citas.crear','citas.editar',
  'productos.ver'
);

-- cajero
INSERT INTO `rol_permisos` (`rol_id`, `permiso_id`, `granted`)
SELECT 6, id, 1 FROM `permisos` WHERE `nombre` IN (
  'dashboard.ver',
  'caja.ver','caja.abrir','caja.cerrar','caja.movimiento',
  'ventas.ver','ventas.crear','ventas.cotizaciones',
  'pacientes.ver'
);

-- ── Empresa demo ─────────────────────────────────────────────
INSERT INTO `empresas` (`id`, `plan_id`, `nombre`, `razon_social`, `rut`, `email`, `telefono`, `direccion`, `ciudad`, `pais`, `moneda`, `color_primario`, `estado`) VALUES
(1, 2, 'Óptica Visión Clara', 'Óptica Visión Clara SpA', '76.123.456-7', 'contacto@visionclaraoptica.cl', '+56 2 2345 6789', 'Av. Providencia 1234, Of. 501', 'Santiago', 'Chile', 'CLP', '#0d6efd', 'activa');

-- ── Licencia demo (1 año) ─────────────────────────────────────
INSERT INTO `licencias` (`empresa_id`, `plan_id`, `tipo`, `fecha_inicio`, `fecha_fin`, `monto`, `estado`) VALUES
(1, 2, 'anual', CURDATE(), DATE_ADD(CURDATE(), INTERVAL 1 YEAR), 599000, 'activa');

-- ── Sucursales demo ───────────────────────────────────────────
INSERT INTO `sucursales` (`id`, `empresa_id`, `nombre`, `codigo`, `direccion`, `telefono`, `email`, `horario`, `es_principal`, `estado`) VALUES
(1, 1, 'Casa Matriz - Providencia', 'SUC-001', 'Av. Providencia 1234', '+56 2 2345 6789', 'providencia@visionclaraoptica.cl', 'Lun-Vie 9:00-19:00, Sáb 10:00-14:00', 1, 'activa'),
(2, 1, 'Sucursal Las Condes',       'SUC-002', 'Av. Apoquindo 4500',   '+56 2 2987 6543', 'lascondes@visionclaraoptica.cl',   'Lun-Vie 9:00-19:00, Sáb 10:00-14:00', 0, 'activa');

-- ── Usuarios demo (contraseña: password) ─────────────────────
INSERT INTO `usuarios` (`id`, `empresa_id`, `sucursal_id`, `rol_id`, `nombre`, `apellido`, `email`, `password`, `es_superadmin`, `estado`) VALUES
(1, NULL, NULL, 1, 'Super',  'Admin',    'superadmin@opticore.com',               '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1, 'activo'),
(2, 1,    1,    2, 'Carlos', 'Mendoza',  'admin@visionclaraoptica.cl',            '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo'),
(3, 1,    1,    3, 'María',  'González', 'jefe.providencia@visionclaraoptica.cl', '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo'),
(4, 1,    1,    4, 'Pedro',  'Ramírez',  'vendedor@visionclaraoptica.cl',         '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo'),
(5, 1,    1,    5, 'Ana',    'Soto',     'optometrista@visionclaraoptica.cl',     '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo'),
(6, 1,    1,    6, 'Luis',   'Herrera',  'cajero@visionclaraoptica.cl',           '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo'),
(7, 1,    2,    3, 'Sofía',  'Vargas',   'jefe.lascondes@visionclaraoptica.cl',   '$2y$12$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'activo');

-- ── Categorías ────────────────────────────────────────────────
INSERT INTO `categorias` (`empresa_id`, `nombre`, `descripcion`, `estado`) VALUES
(1, 'Marcos',             'Armazones y monturas',                 'activa'),
(1, 'Lentes Oftálmicos',  'Lentes de prescripción',               'activa'),
(1, 'Lentes de Contacto', 'Lentes de contacto blandos y rígidos', 'activa'),
(1, 'Soluciones',         'Soluciones para lentes de contacto',   'activa'),
(1, 'Accesorios',         'Estuches, cordones, paños, etc.',      'activa'),
(1, 'Servicios',          'Servicios ópticos',                    'activa');

-- ── Marcas ────────────────────────────────────────────────────
INSERT INTO `marcas` (`empresa_id`, `nombre`, `descripcion`, `estado`) VALUES
(1, 'Ray-Ban',     'Marca premium de gafas',          'activa'),
(1, 'Oakley',      'Gafas deportivas y premium',      'activa'),
(1, 'Silhouette',  'Marcos austríacos de alta gama',  'activa'),
(1, 'Essilor',     'Lentes oftálmicos premium',       'activa'),
(1, 'Hoya',        'Lentes de alta tecnología',       'activa'),
(1, 'Acuvue',      'Lentes de contacto Johnson',      'activa'),
(1, 'Bausch+Lomb', 'Lentes de contacto y soluciones', 'activa'),
(1, 'Sin Marca',   'Productos genéricos',             'activa');

-- ── Proveedores ───────────────────────────────────────────────
INSERT INTO `proveedores` (`empresa_id`, `nombre`, `rut`, `contacto`, `telefono`, `email`, `tipo`, `estado`) VALUES
(1, 'Distribuidora Óptica Sur', '77.234.567-8', 'Roberto Pérez', '+56 2 2111 2222', 'ventas@opticasur.cl', 'proveedor',   'activo'),
(1, 'Laboratorio Visión Total', '76.345.678-9', 'Carmen López',  '+56 2 2333 4444', 'lab@visiontotal.cl',  'laboratorio', 'activo'),
(1, 'Essilor Chile',            '76.456.789-0', 'Diego Morales', '+56 2 2555 6666', 'chile@essilor.com',   'ambos',       'activo');

-- ── Formas de pago ────────────────────────────────────────────
INSERT INTO `formas_pago` (`empresa_id`, `nombre`, `tipo`, `estado`) VALUES
(1, 'Efectivo',               'efectivo',      'activa'),
(1, 'Tarjeta de Débito',      'tarjeta',       'activa'),
(1, 'Tarjeta de Crédito',     'tarjeta',       'activa'),
(1, 'Transferencia Bancaria', 'transferencia', 'activa'),
(1, 'Crédito Interno',        'credito',       'activa');

-- ── Productos ─────────────────────────────────────────────────
INSERT INTO `productos` (`empresa_id`, `categoria_id`, `marca_id`, `proveedor_id`, `sku`, `nombre`, `tipo`, `precio_costo`, `precio_venta`, `impuesto`, `stock_minimo`, `estado`) VALUES
(1, 1, 1, 1, 'RB-3025-001',  'Ray-Ban Aviator RB3025 Dorado',         'marco',     45000, 89990, 0,  3, 'activo'),
(1, 1, 1, 1, 'RB-2140-001',  'Ray-Ban Wayfarer RB2140 Negro',         'marco',     42000, 79990, 0,  3, 'activo'),
(1, 1, 2, 1, 'OAK-OO9102',   'Oakley Holbrook Matte Black',           'marco',     55000, 99990, 0,  2, 'activo'),
(1, 1, 3, 1, 'SIL-5515',     'Silhouette Titan Minimal Art 5515',     'marco',     85000,159990, 0,  2, 'activo'),
(1, 1, 8, 1, 'GEN-METAL-01', 'Marco Metálico Genérico Plateado',      'marco',      8000, 19990, 0,  5, 'activo'),
(1, 2, 4, 3, 'ESS-VARILUX',  'Lente Varilux Comfort 3.0 Progresivo',  'lente',     35000, 69990, 0,  0, 'activo'),
(1, 2, 5, 1, 'HOY-NULUX',    'Lente Hoya Nulux 1.60 AR',              'lente',     28000, 54990, 0,  0, 'activo'),
(1, 2, 8, 1, 'GEN-LENTE-01', 'Lente Orgánico 1.50 Antireflejo',       'lente',     12000, 24990, 0,  0, 'activo'),
(1, 3, 6, 1, 'ACU-1DAY-30',  'Acuvue 1-Day Moist 30 unidades',        'lente',      8500, 18990, 0,  5, 'activo'),
(1, 3, 7, 1, 'BL-ULTRA-6',   'Bausch+Lomb Ultra 6 unidades',          'lente',     12000, 24990, 0,  5, 'activo'),
(1, 4, 7, 1, 'BL-SOL-360',   'Bausch+Lomb Solución 360ml',            'solucion',   4500,  9990, 0,  8, 'activo'),
(1, 5, 8, 1, 'ACC-ESTUCHE',  'Estuche Rígido Universal',              'accesorio',  1500,  3990, 0, 10, 'activo'),
(1, 5, 8, 1, 'ACC-PANO',     'Paño Microfibra Limpieza',              'accesorio',   500,  1990, 0, 20, 'activo'),
(1, 6, 8, 1, 'SRV-EXAM',     'Examen Visual Completo',                'servicio',      0, 15000, 0,  0, 'activo'),
(1, 6, 8, 1, 'SRV-AJUSTE',   'Ajuste y Reparación de Marcos',         'servicio',      0,  5000, 0,  0, 'activo');

-- ── Stock ─────────────────────────────────────────────────────
INSERT INTO `stock` (`empresa_id`, `sucursal_id`, `producto_id`, `cantidad`) VALUES
(1,1, 1,5),(1,1, 2,8),(1,1, 3,4),(1,1, 4,2),(1,1, 5,12),
(1,1, 6,0),(1,1, 7,0),(1,1, 8,0),(1,1, 9,15),(1,1,10,8),
(1,1,11,10),(1,1,12,20),(1,1,13,30),(1,1,14,0),(1,1,15,0),
(1,2, 1,3),(1,2, 2,5),(1,2, 3,2),(1,2, 4,1),(1,2, 5,8),
(1,2, 6,0),(1,2, 7,0),(1,2, 8,0),(1,2, 9,10),(1,2,10,5),
(1,2,11,6),(1,2,12,15),(1,2,13,20),(1,2,14,0),(1,2,15,0);

-- ── Pacientes demo ────────────────────────────────────────────
INSERT INTO `pacientes` (`empresa_id`, `sucursal_id`, `nombre`, `apellido`, `rut`, `fecha_nacimiento`, `genero`, `telefono`, `email`, `ciudad`, `estado`) VALUES
(1,1,'Juan',     'Pérez Silva',    '12.345.678-9','1985-03-15','M','+56 9 8765 4321','juan.perez@gmail.com',   'Santiago',   'activo'),
(1,1,'María',    'López Rojas',    '13.456.789-0','1990-07-22','F','+56 9 7654 3210','maria.lopez@gmail.com',  'Santiago',   'activo'),
(1,1,'Carlos',   'Martínez Díaz',  '14.567.890-1','1978-11-08','M','+56 9 6543 2109','carlos.m@hotmail.com',   'Providencia','activo'),
(1,1,'Ana',      'García Muñoz',   '15.678.901-2','1995-04-30','F','+56 9 5432 1098','ana.garcia@gmail.com',   'Las Condes', 'activo'),
(1,2,'Roberto',  'Sánchez Torres', '16.789.012-3','1982-09-12','M','+56 9 4321 0987','roberto.s@yahoo.com',    'Vitacura',   'activo'),
(1,2,'Claudia',  'Fernández Vega', '17.890.123-4','1988-01-25','F','+56 9 3210 9876','claudia.f@gmail.com',    'Las Condes', 'activo'),
