<?php
/**
 * OptiCore SaaS - Auth Model
 * Maneja autenticación, sesión y permisos
 */

class Auth
{
    // ── Login ─────────────────────────────────────────────────
    public static function attempt(string $email, string $password): bool
    {
        $db   = db();
        $user = $db->fetchOne(
            "SELECT u.*, r.slug as rol_slug, r.nombre as rol_nombre,
                    e.nombre as empresa_nombre, e.estado as empresa_estado,
                    e.moneda as empresa_moneda,
                    s.nombre as sucursal_nombre
             FROM usuarios u
             LEFT JOIN roles r ON u.rol_id = r.id
             LEFT JOIN empresas e ON u.empresa_id = e.id
             LEFT JOIN sucursales s ON u.sucursal_id = s.id
             WHERE u.email = ? AND u.estado = 'activo'
             LIMIT 1",
            [$email]
        );

        if (!$user || !password_verify($password, $user['password'])) {
            return false;
        }

        // Verificar empresa activa (excepto superadmin)
        if (!$user['es_superadmin'] && $user['empresa_estado'] !== 'activa') {
            return false;
        }

        // Cargar permisos del rol
        $permisos = $db->fetchAll(
            "SELECT p.nombre FROM rol_permisos rp
             JOIN permisos p ON rp.permiso_id = p.id
             WHERE rp.rol_id = ? AND rp.granted = 1",
            [$user['rol_id']]
        );
        $permisosArray = array_column($permisos, 'nombre');

        // Guardar en sesión
        $_SESSION['user'] = [
            'id'               => $user['id'],
            'nombre'           => $user['nombre'],
            'apellido'         => $user['apellido'],
            'email'            => $user['email'],
            'empresa_id'       => $user['empresa_id'],
            'empresa_nombre'   => $user['empresa_nombre'],
            'empresa_moneda'   => $user['empresa_moneda'] ?? 'CLP',
            'sucursal_id'      => $user['sucursal_id'],
            'sucursal_nombre'  => $user['sucursal_nombre'],
            'rol_id'           => $user['rol_id'],
            'rol_slug'         => $user['rol_slug'],
            'rol_nombre'       => $user['rol_nombre'],
            'es_superadmin'    => (bool) $user['es_superadmin'],
            'permisos'         => $permisosArray,
            'avatar'           => $user['avatar'],
        ];

        // Regenerar ID de sesión (seguridad)
        session_regenerate_id(true);

        // Actualizar último login
        $db->update('usuarios', ['ultimo_login' => date('Y-m-d H:i:s')], ['id' => $user['id']]);

        // Audit log
        AuditLog::log('login', 'usuarios', $user['id']);

        return true;
    }

    // ── Logout ────────────────────────────────────────────────
    public static function logout(): void
    {
        if (self::check()) {
            AuditLog::log('logout', 'usuarios', self::id());
        }
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params['path'], $params['domain'],
                $params['secure'], $params['httponly']
            );
        }
        session_destroy();
    }

    // ── Verificar si está autenticado ─────────────────────────
    public static function check(): bool
    {
        return isset($_SESSION['user']['id']);
    }

    // ── Obtener usuario actual ────────────────────────────────
    public static function user(): ?array
    {
        return $_SESSION['user'] ?? null;
    }

    // ── Obtener campo del usuario ─────────────────────────────
    public static function id(): ?int
    {
        return $_SESSION['user']['id'] ?? null;
    }

    public static function empresaId(): ?int
    {
        return $_SESSION['user']['empresa_id'] ?? null;
    }

    public static function sucursalId(): ?int
    {
        return $_SESSION['user']['sucursal_id'] ?? null;
    }

    public static function rolSlug(): ?string
    {
        return $_SESSION['user']['rol_slug'] ?? null;
    }

    public static function isSuperAdmin(): bool
    {
        return (bool) ($_SESSION['user']['es_superadmin'] ?? false);
    }

    // ── Verificar permiso ─────────────────────────────────────
    public static function hasPermission(string $permiso): bool
    {
        if (!self::check()) return false;
        if (self::isSuperAdmin()) return true;
        return in_array($permiso, $_SESSION['user']['permisos'] ?? [], true);
    }

    // ── Verificar múltiples permisos (cualquiera) ─────────────
    public static function hasAnyPermission(array $permisos): bool
    {
        foreach ($permisos as $p) {
            if (self::hasPermission($p)) return true;
        }
        return false;
    }

    // ── Verificar múltiples permisos (todos) ──────────────────
    public static function hasAllPermissions(array $permisos): bool
    {
        foreach ($permisos as $p) {
            if (!self::hasPermission($p)) return false;
        }
        return true;
    }

    // ── Requerir autenticación ────────────────────────────────
    public static function requireAuth(): void
    {
        if (!self::check()) {
            flash('error', 'Debes iniciar sesión para acceder.');
            redirect('/login');
        }
    }

    // ── Requerir permiso ──────────────────────────────────────
    public static function requirePermission(string $permiso): void
    {
        self::requireAuth();
        if (!self::hasPermission($permiso)) {
            http_response_code(403);
            if (isAjax()) {
                jsonResponse(['error' => 'Sin permisos para esta acción'], 403);
            }
            flash('error', 'No tienes permisos para realizar esta acción.');
            redirect('/');
        }
    }

    // ── Requerir superadmin ───────────────────────────────────
    public static function requireSuperAdmin(): void
    {
        self::requireAuth();
        if (!self::isSuperAdmin()) {
            http_response_code(403);
            flash('error', 'Acceso restringido a Super Administradores.');
            redirect('/');
        }
    }

    // ── Refrescar permisos en sesión ──────────────────────────
    public static function refreshPermissions(): void
    {
        if (!self::check()) return;
        $db = db();
        $permisos = $db->fetchAll(
            "SELECT p.nombre FROM rol_permisos rp
             JOIN permisos p ON rp.permiso_id = p.id
             WHERE rp.rol_id = ? AND rp.granted = 1",
            [$_SESSION['user']['rol_id']]
        );
        $_SESSION['user']['permisos'] = array_column($permisos, 'nombre');
    }

    // ── Moneda de la empresa ──────────────────────────────────
    public static function moneda(): string
    {
        return $_SESSION['user']['empresa_moneda'] ?? 'CLP';
    }
}
