<?php
/**
 * OptiCore SaaS - RecetaController
 */

class RecetaController
{
    private Receta $model;

    public function __construct()
    {
        Auth::requirePermission('recetas.ver');
        $this->model = new Receta();
    }

    // ── GET /recetas ──────────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $pacienteId = getInt('paciente_id');
        $resultado  = $this->model->listar($page, ITEMS_PER_PAGE, $pacienteId ?: null);

        view('recetas.index', [
            'title'      => 'Recetas',
            'recetas'    => $resultado['data'],
            'pagination' => $resultado['pagination'],
            'paciente_id'=> $pacienteId,
        ]);
    }

    // ── GET /recetas/create ───────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('recetas.crear');

        $pacienteId          = getInt('paciente_id');
        $pacienteSeleccionado = null;
        if ($pacienteId) {
            $pacienteSeleccionado = (new Paciente())->findById($pacienteId);
        }

        view('recetas.create', [
            'title'                => 'Nueva Receta',
            'pacienteSeleccionado' => $pacienteSeleccionado,
        ]);
    }

    // ── POST /recetas ─────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('recetas.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'paciente_id' => 'Paciente',
            'fecha'       => 'Fecha',
            'tipo'        => 'Tipo',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/recetas/create');
        }

        // Subir adjunto si existe
        $adjunto = null;
        if (!empty($_FILES['adjunto']['name'])) {
            $adjunto = uploadFile($_FILES['adjunto'], 'recetas', ['jpg','jpeg','png','pdf']);
        }

        $id = $this->model->crear(array_merge($_POST, [
            'usuario_id' => Auth::id(),
            'adjunto'    => $adjunto,
        ]));

        AuditLog::log('crear_receta', 'recetas', $id);
        flash('success', 'Receta registrada correctamente.');
        redirect("/recetas/$id");
    }

    // ── GET /recetas/{id} ─────────────────────────────────────
    public function show(int $id): void
    {
        $receta = $this->model->getById($id);
        if (!$receta) {
            flash('error', 'Receta no encontrada.');
            redirect('/recetas');
        }

        view('recetas.show', [
            'title'  => 'Receta #' . $id,
            'receta' => $receta,
        ]);
    }

    // ── GET /recetas/{id}/print ───────────────────────────────
    public function print(int $id): void
    {
        Auth::requirePermission('recetas.imprimir');

        $receta = $this->model->getById($id);
        if (!$receta) {
            flash('error', 'Receta no encontrada.');
            redirect('/recetas');
        }

        $empresa = db()->fetchOne(
            "SELECT * FROM empresas WHERE id = ?",
            [Auth::empresaId() ?? $receta['empresa_id']]
        );

        view('recetas.print', [
            'title'   => 'Imprimir Receta',
            'receta'  => $receta,
            'empresa' => $empresa,
        ], 'print');
    }

    // ── POST /recetas/{id}/delete ─────────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('recetas.eliminar');
        csrf_verify();

        $receta = $this->model->getById($id);
        if (!$receta) {
            flash('error', 'Receta no encontrada.');
            redirect('/recetas');
        }

        $this->model->anular($id);
        AuditLog::log('anular_receta', 'recetas', $id, $receta);
        flash('success', 'Receta anulada correctamente.');
        redirect('/recetas');
    }
}
