<?php
/**
 * OptiCore SaaS - PlanController (Solo SuperAdmin)
 */

class PlanController
{
    private Plan $model;

    public function __construct()
    {
        Auth::requireSuperAdmin();
        $this->model = new Plan();
    }

    // ── GET /planes ───────────────────────────────────────────
    public function index(): void
    {
        $planes = $this->model->getAll();
        view('planes.index', ['title' => 'Planes', 'planes' => $planes]);
    }

    // ── GET /planes/create ────────────────────────────────────
    public function create(): void
    {
        view('planes.create', ['title' => 'Nuevo Plan']);
    }

    // ── POST /planes ──────────────────────────────────────────
    public function store(): void
    {
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre'         => 'Nombre',
            'precio_mensual' => 'Precio mensual',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/planes/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_plan', 'planes', $id, null, $_POST);
        flash('success', 'Plan creado correctamente.');
        redirect('/planes');
    }

    // ── GET /planes/{id}/edit ─────────────────────────────────
    public function edit(int $id): void
    {
        $plan = $this->model->getById($id);
        if (!$plan) {
            flash('error', 'Plan no encontrado.');
            redirect('/planes');
        }
        view('planes.edit', ['title' => 'Editar Plan', 'plan' => $plan]);
    }

    // ── POST /planes/{id} ─────────────────────────────────────
    public function update(int $id): void
    {
        csrf_verify();

        $plan = $this->model->getById($id);
        if (!$plan) {
            flash('error', 'Plan no encontrado.');
            redirect('/planes');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/planes/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_plan', 'planes', $id, $plan, $_POST);
        flash('success', 'Plan actualizado correctamente.');
        redirect('/planes');
    }

    // ── POST /planes/{id}/delete ──────────────────────────────
    public function destroy(int $id): void
    {
        csrf_verify();

        $plan = $this->model->getById($id);
        if (!$plan) {
            flash('error', 'Plan no encontrado.');
            redirect('/planes');
        }

        if (!$this->model->eliminar($id)) {
            flash('error', 'No se puede eliminar el plan porque tiene empresas asociadas.');
            redirect('/planes');
        }

        AuditLog::log('eliminar_plan', 'planes', $id, $plan);
        flash('success', 'Plan eliminado correctamente.');
        redirect('/planes');
    }
}
