<?php
/**
 * OptiCore SaaS - PerfilController
 * Perfil del usuario autenticado
 */

class PerfilController
{
    public function __construct()
    {
        Auth::requireAuth();
    }

    // ── GET /perfil ───────────────────────────────────────────
    public function index(): void
    {
        $usuario = db()->fetchOne(
            "SELECT u.*, r.nombre as rol_nombre, s.nombre as sucursal_nombre, e.nombre as empresa_nombre
             FROM usuarios u
             LEFT JOIN roles r      ON u.rol_id      = r.id
             LEFT JOIN sucursales s ON u.sucursal_id = s.id
             LEFT JOIN empresas e   ON u.empresa_id  = e.id
             WHERE u.id = ?",
            [Auth::id()]
        );

        view('perfil.index', [
            'title'   => 'Mi Perfil',
            'usuario' => $usuario,
        ]);
    }

    // ── POST /perfil ──────────────────────────────────────────
    public function update(): void
    {
        Auth::requireAuth();
        csrf_verify();

        $id      = Auth::id();
        $usuario = db()->fetchOne("SELECT * FROM usuarios WHERE id = ?", [$id]);

        if (!$usuario) {
            flash('error', 'Usuario no encontrado.');
            redirect('/perfil');
        }

        $errors = validateRequired($_POST, [
            'nombre'   => 'Nombre',
            'apellido' => 'Apellido',
        ]);

        if (!empty($_POST['email']) && !filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'El email no es válido.';
        }

        // Verificar email único (si cambió)
        if (!empty($_POST['email']) && $_POST['email'] !== $usuario['email']) {
            $existe = db()->fetchColumn(
                "SELECT COUNT(*) FROM usuarios WHERE email = ? AND id != ?",
                [$_POST['email'], $id]
            );
            if ($existe) {
                $errors['email'] = 'El email ya está en uso por otro usuario.';
            }
        }

        // Validar contraseña si se quiere cambiar
        if (!empty($_POST['password_nueva'])) {
            if (strlen($_POST['password_nueva']) < 8) {
                $errors['password_nueva'] = 'La contraseña debe tener al menos 8 caracteres.';
            }
            if ($_POST['password_nueva'] !== ($_POST['password_confirmar'] ?? '')) {
                $errors['password_confirmar'] = 'Las contraseñas no coinciden.';
            }
            if (!password_verify($_POST['password_actual'] ?? '', $usuario['password'])) {
                $errors['password_actual'] = 'La contraseña actual es incorrecta.';
            }
        }

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/perfil');
        }

        // Subir avatar si se envió
        $avatar = $usuario['avatar'];
        if (!empty($_FILES['avatar']['name'])) {
            $uploaded = uploadFile($_FILES['avatar'], 'avatars', ['jpg', 'jpeg', 'png', 'gif', 'webp']);
            if ($uploaded) {
                $avatar = $uploaded;
            }
        }

        $campos = [
            'nombre'   => trim($_POST['nombre']),
            'apellido' => trim($_POST['apellido']),
            'telefono' => trim($_POST['telefono'] ?? '') ?: null,
            'avatar'   => $avatar,
        ];

        // Solo actualizar email si se proporcionó
        if (!empty($_POST['email'])) {
            $campos['email'] = trim($_POST['email']);
        }

        // Actualizar contraseña si se proporcionó
        if (!empty($_POST['password_nueva'])) {
            $campos['password'] = password_hash($_POST['password_nueva'], PASSWORD_BCRYPT, ['cost' => BCRYPT_COST]);
        }

        db()->update('usuarios', $campos, ['id' => $id]);

        // Actualizar datos en sesión
        $_SESSION['user']['nombre']  = $campos['nombre'];
        $_SESSION['user']['apellido'] = $campos['apellido'];
        if (isset($campos['email'])) {
            $_SESSION['user']['email'] = $campos['email'];
        }
        if ($avatar !== $usuario['avatar']) {
            $_SESSION['user']['avatar'] = $avatar;
        }

        AuditLog::log('editar_perfil', 'usuarios', $id);
        flash('success', 'Perfil actualizado correctamente.');
        redirect('/perfil');
    }
}
