<?php
/**
 * OptiCore SaaS - CategoriaController
 */

class CategoriaController
{
    private Categoria $model;

    public function __construct()
    {
        Auth::requirePermission('categorias.ver');
        $this->model = new Categoria();
    }

    // ── GET /categorias ───────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $resultado = $this->model->getAll($page, $busqueda);

        view('categorias.index', [
            'title'       => 'Categorías',
            'categorias'  => $resultado['data'],
            'pagination'  => $resultado,
            'busqueda'    => $busqueda,
        ]);
    }

    // ── GET /categorias/create ────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('categorias.crear');
        view('categorias.create', ['title' => 'Nueva Categoría']);
    }

    // ── POST /categorias ──────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('categorias.crear');
        csrf_verify();

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/categorias/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_categoria', 'categorias', $id);
        flash('success', 'Categoría creada correctamente.');
        redirect('/categorias');
    }

    // ── GET /categorias/{id}/edit ─────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('categorias.editar');

        $categoria = $this->model->getById($id);
        if (!$categoria) {
            flash('error', 'Categoría no encontrada.');
            redirect('/categorias');
        }

        view('categorias.edit', [
            'title'     => 'Editar Categoría',
            'categoria' => $categoria,
        ]);
    }

    // ── POST /categorias/{id} ─────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('categorias.editar');
        csrf_verify();

        $categoria = $this->model->getById($id);
        if (!$categoria) {
            flash('error', 'Categoría no encontrada.');
            redirect('/categorias');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/categorias/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_categoria', 'categorias', $id, $categoria, $_POST);
        flash('success', 'Categoría actualizada correctamente.');
        redirect('/categorias');
    }

    // ── POST /categorias/{id}/delete ──────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('categorias.eliminar');
        csrf_verify();

        $categoria = $this->model->getById($id);
        if (!$categoria) {
            flash('error', 'Categoría no encontrada.');
            redirect('/categorias');
        }

        if ($this->model->tieneProductos($id)) {
            flash('error', 'No se puede eliminar: la categoría tiene productos asociados.');
            redirect('/categorias');
        }

        $this->model->eliminar($id);
        AuditLog::log('eliminar_categoria', 'categorias', $id, $categoria);
        flash('success', 'Categoría desactivada correctamente.');
        redirect('/categorias');
    }
}
