<?php
/**
 * OptiCore SaaS - AuditLogController
 * Vista de logs de auditoría para administradores
 */

class AuditLogController
{
    public function __construct()
    {
        Auth::requirePermission('audit_logs.ver');
    }

    // ── GET /audit-logs ───────────────────────────────────────
    public function index(): void
    {
        $empresaId = Auth::empresaId();
        $db        = db();

        // Filtros
        $usuarioId = getInt('usuario_id') ?: null;
        $accion    = get('accion') ?: null;
        $tabla     = get('tabla') ?: null;
        $desde     = get('desde') ?: date('Y-m-01');
        $hasta     = get('hasta') ?: date('Y-m-d');
        $page      = currentPage();
        $perPage   = 50;

        // Construir WHERE dinámico
        $where  = ['al.empresa_id = ?'];
        $params = [$empresaId];

        if ($usuarioId) {
            $where[]  = 'al.usuario_id = ?';
            $params[] = $usuarioId;
        }
        if ($accion) {
            $where[]  = 'al.accion LIKE ?';
            $params[] = '%' . $accion . '%';
        }
        if ($tabla) {
            $where[]  = 'al.tabla = ?';
            $params[] = $tabla;
        }
        if ($desde) {
            $where[]  = 'DATE(al.fecha) >= ?';
            $params[] = $desde;
        }
        if ($hasta) {
            $where[]  = 'DATE(al.fecha) <= ?';
            $params[] = $hasta;
        }

        $whereStr = implode(' AND ', $where);

        // Total para paginación
        $total = $db->fetchOne(
            "SELECT COUNT(*) as c FROM audit_logs al WHERE $whereStr",
            $params
        )['c'] ?? 0;

        $totalPages = (int) ceil($total / $perPage);
        $offset     = ($page - 1) * $perPage;

        // Logs paginados
        $logs = $db->fetchAll(
            "SELECT al.*,
                    CONCAT(u.nombre, ' ', u.apellido) as usuario_nombre,
                    u.email as usuario_email
             FROM audit_logs al
             LEFT JOIN usuarios u ON al.usuario_id = u.id
             WHERE $whereStr
             ORDER BY al.fecha DESC
             LIMIT $perPage OFFSET $offset",
            $params
        );

        // Usuarios para filtro
        $usuarios = $db->fetchAll(
            "SELECT DISTINCT u.id, CONCAT(u.nombre,' ',u.apellido) as nombre
             FROM audit_logs al
             JOIN usuarios u ON al.usuario_id = u.id
             WHERE al.empresa_id = ?
             ORDER BY nombre",
            [$empresaId]
        );

        // Tablas únicas para filtro
        $tablas = $db->fetchAll(
            "SELECT DISTINCT tabla FROM audit_logs
             WHERE empresa_id = ? AND tabla IS NOT NULL AND tabla != ''
             ORDER BY tabla",
            [$empresaId]
        );
        $tablas = array_column($tablas, 'tabla');

        // Acciones únicas para filtro
        $acciones = $db->fetchAll(
            "SELECT DISTINCT accion FROM audit_logs
             WHERE empresa_id = ?
             ORDER BY accion",
            [$empresaId]
        );
        $acciones = array_column($acciones, 'accion');

        $pagination = [
            'current_page' => $page,
            'total_pages'  => $totalPages,
            'total'        => $total,
            'per_page'     => $perPage,
        ];

        view('audit_logs.index', [
            'title'      => 'Audit Logs',
            'logs'       => $logs,
            'usuarios'   => $usuarios,
            'tablas'     => $tablas,
            'acciones'   => $acciones,
            'pagination' => $pagination,
            'filtros'    => compact('usuarioId', 'accion', 'tabla', 'desde', 'hasta'),
        ]);
    }
}
