<?php
/**
 * OptiCore SaaS - AgendaController
 * Gestión de Citas / Agenda
 */

class AgendaController
{
    private Cita $model;

    public function __construct()
    {
        Auth::requirePermission('agenda.ver');
        $this->model = new Cita();
    }

    // ── GET /agenda ───────────────────────────────────────────
    public function index(): void
    {
        $page    = currentPage();
        $filtros = [
            'desde'  => get('desde') ?: date('Y-m-d'),
            'hasta'  => get('hasta') ?: date('Y-m-d', strtotime('+7 days')),
            'estado' => get('estado'),
            'q'      => get('q'),
        ];

        $resultado = $this->model->getAll($page, array_filter($filtros, fn($v) => $v !== ''));
        $contadores = $this->model->contarPorEstado(date('Y-m-d'));

        view('agenda.index', [
            'title'      => 'Agenda',
            'citas'      => $resultado['data'],
            'pagination' => $resultado,
            'filtros'    => $filtros,
            'contadores' => $contadores,
        ]);
    }

    // ── GET /agenda/create ────────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('agenda.crear');

        $fecha = get('fecha') ?: date('Y-m-d');

        view('agenda.create', [
            'title' => 'Nueva Cita',
            'fecha' => $fecha,
        ]);
    }

    // ── POST /agenda ──────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('agenda.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'fecha' => 'Fecha',
            'hora'  => 'Hora',
        ]);

        // Requiere nombre de paciente si no se seleccionó uno existente
        if (empty($_POST['paciente_id']) && empty(trim($_POST['nombre_paciente'] ?? ''))) {
            $errors['nombre_paciente'] = 'Nombre del paciente es requerido.';
        }

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/agenda/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_cita', 'citas', $id);
        flash('success', 'Cita agendada correctamente.');
        redirect('/agenda');
    }

    // ── GET /agenda/{id}/edit ─────────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('agenda.editar');

        $cita = $this->model->getById($id);
        if (!$cita) {
            flash('error', 'Cita no encontrada.');
            redirect('/agenda');
        }

        view('agenda.edit', [
            'title' => 'Editar Cita',
            'cita'  => $cita,
        ]);
    }

    // ── POST /agenda/{id} ─────────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('agenda.editar');
        csrf_verify();

        $cita = $this->model->getById($id);
        if (!$cita) {
            flash('error', 'Cita no encontrada.');
            redirect('/agenda');
        }

        $errors = validateRequired($_POST, [
            'fecha' => 'Fecha',
            'hora'  => 'Hora',
        ]);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/agenda/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_cita', 'citas', $id, $cita, $_POST);
        flash('success', 'Cita actualizada correctamente.');
        redirect('/agenda');
    }

    // ── POST /agenda/{id}/estado ──────────────────────────────
    public function cambiarEstado(int $id): void
    {
        Auth::requirePermission('agenda.editar');
        csrf_verify();

        $cita = $this->model->getById($id);
        if (!$cita) {
            if (isAjax()) jsonResponse(['error' => 'Cita no encontrada'], 404);
            flash('error', 'Cita no encontrada.');
            redirect('/agenda');
        }

        $estado = $_POST['estado'] ?? '';
        $estadosValidos = ['pendiente', 'confirmada', 'atendida', 'cancelada', 'no_asistio'];
        if (!in_array($estado, $estadosValidos)) {
            if (isAjax()) jsonResponse(['error' => 'Estado no válido'], 422);
            flash('error', 'Estado no válido.');
            redirect('/agenda');
        }

        $this->model->cambiarEstado($id, $estado);
        AuditLog::log('cambiar_estado_cita', 'citas', $id, $cita, ['estado' => $estado]);

        if (isAjax()) {
            jsonResponse(['success' => true, 'estado' => $estado]);
        }

        flash('success', 'Estado de cita actualizado.');
        redirect('/agenda');
    }

    // ── POST /agenda/{id}/delete ──────────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('agenda.eliminar');
        csrf_verify();

        $cita = $this->model->getById($id);
        if (!$cita) {
            flash('error', 'Cita no encontrada.');
            redirect('/agenda');
        }

        $this->model->delete($id);
        AuditLog::log('eliminar_cita', 'citas', $id, $cita);
        flash('success', 'Cita eliminada correctamente.');
        redirect('/agenda');
    }
}
